unit WinExam1;

{
  Windows user-interface to examination objects.
  Requires MSXML v3 package from Microsoft.

  Copyright  Keith Wood (kbwood@iprimus.com.au)
  Version 1.0 - 22 October, 1999.
}

interface

uses
  Windows, Messages, SysUtils, Classes, Graphics, Controls, Forms, Dialogs,
  StdCtrls, ExtCtrls, Buttons, ComCtrls, Exams;

type
  TfrmExam = class(TForm)
    pgcExam: TPageControl;
      tshIntro: TTabSheet;
        lblInitLetter: TStaticText;
        lblTitle: TStaticText;
        memDescription: TMemo;
        Label1: TLabel;
        lblCount: TLabel;
        Label3: TLabel;
        lblPass: TLabel;
        Label2: TLabel;
        memInstructions: TMemo;
        btnStart: TBitBtn;
      tshQuestion: TTabSheet;
        pnlQuery: TPanel;
          StaticText1: TStaticText;
          StaticText2: TStaticText;
          memQuestion: TMemo;
          lblQuestion: TStaticText;
        pnlAnswers: TPanel;
          StaticText3: TStaticText;
          StaticText4: TStaticText;
          pgcAnswers: TPageControl;
            tshCheckbox: TTabSheet;
              imgCheckBox1: TImage;
              cbxAnswer1: TCheckBox;
              imgCheckBox2: TImage;
              cbxAnswer2: TCheckBox;
              imgCheckBox3: TImage;
              cbxAnswer3: TCheckBox;
              imgCheckBox4: TImage;
              cbxAnswer4: TCheckBox;
              imgCheckBox5: TImage;
              cbxAnswer5: TCheckBox;
            tshRadio: TTabSheet;
              imgRadio1: TImage;
              rabAnswer1: TRadioButton;
              imgRadio2: TImage;
              rabAnswer2: TRadioButton;
              imgRadio3: TImage;
              rabAnswer3: TRadioButton;
              imgRadio4: TImage;
              rabAnswer4: TRadioButton;
              imgRadio5: TImage;
              rabAnswer5: TRadioButton;
            tshText: TTabSheet;
              imgText: TImage;
              edtAnswer: TEdit;
        pnlExplanation: TPanel;
          StaticText5: TStaticText;
          StaticText6: TStaticText;
          memExplanation: TMemo;
        pnlButtons: TPanel;
          btnPrevious: TBitBtn;
          btnNext: TBitBtn;
          btnScore: TBitBtn;
      tshScore: TTabSheet;
        lblScoreInitLetter: TStaticText;
        lblScoreTitle: TStaticText;
        lbxQuestions: TListBox;
        Label4: TLabel;
        lblScorePass: TLabel;
        Label6: TLabel;
        Label5: TLabel;
        lblScore: TLabel;
        Label12: TLabel;
        Label7: TLabel;
        lblGrade: TLabel;
        Label8: TLabel;
        lblCorrect: TLabel;
        Label10: TLabel;
        Label11: TLabel;
        lblIncorrect: TLabel;
        Label13: TLabel;
        Label14: TLabel;
        lblUnanswered: TLabel;
        Label9: TLabel;
        btnClose: TBitBtn;
    imgCorrect: TImage;
    imgIncorrect: TImage;
    procedure FormCreate(Sender: TObject);
    procedure lbxQuestionsDblClick(Sender: TObject);
    procedure lbxQuestionsDrawItem(Control: TWinControl; Index: Integer;
      Rect: TRect; State: TOwnerDrawState);
    procedure btnNextPreviousClick(Sender: TObject);
    procedure btnScoreClick(Sender: TObject);
    procedure lbxQuestionsKeyDown(Sender: TObject; var Key: Word;
      Shift: TShiftState);
  private
    FCurQuestion: Integer;
    FMaxAnswers: Integer;
    FCorrect: array [Boolean] of TIcon;
    FExam: TExam;
    FSession: TUserSession;
    procedure InitTest(FileName: string);
    procedure ProcessCurQuestion;
    procedure ProcessQuestion(Question: TQuestion);
    procedure ShowCurQuestion;
    procedure ShowQuestion(Question: TQuestion;
      CurQuestion: Integer; Response: string);
    procedure ShowScore;
  public
  end;

var
  frmExam: TfrmExam;

implementation

{$R *.DFM}

resourcestring
  Accepted = 'Accepted:';
  Failed   = 'Failed';
  NoTest   = 'No test has been specified to be run.';
  Passed   = 'Passed';
  Question = 'Question';

const
  Correct    = 1;
  Incorrect  = 0;
  Unanswered = 2;
  Highlight: array [Boolean] of TColor = (clRed, clGreen);

{ Load resources and initialise test }
procedure TfrmExam.FormCreate(Sender: TObject);
begin
  FCorrect[False]         := imgIncorrect.Picture.Icon;
  FCorrect[True]          := imgCorrect.Picture.Icon;
  FMaxAnswers             := 5;
  FExam                   := nil;
  FSession                := nil;
  lblCorrect.Font.Color   := Highlight[True];
  lblIncorrect.Font.Color := Highlight[False];

  { Load the specified test }
  if ParamCount = 0 then
  begin
    MessageDlg(NoTest, mtError, [mbOK], 0);
    Halt;
  end;
  InitTest(ParamStr(1));
end;

{ Go to the next/previous question }
procedure TfrmExam.btnNextPreviousClick(Sender: TObject);
begin
  ProcessCurQuestion;
  FCurQuestion := FCurQuestion + TBitBtn(Sender).Tag;
  if FCurQuestion > FExam.QuestionCount - 1 then
    ShowScore
  else
    ShowCurQuestion;
end;

{ Go to the scoring page }
procedure TfrmExam.btnScoreClick(Sender: TObject);
begin
  ProcessCurQuestion;
  ShowScore;
end;

{ Load the test and initialise a new user }
procedure TfrmExam.InitTest(FileName: string);
begin
  { Load the test and initialise a new user }
  FExam    := LoadExam(FileName);
  FSession := TUserSession.Create(FExam);
  FSession.InitialiseQuestions;

  Caption            := FExam.Title;
  pgcExam.ActivePage := tshIntro;
  { Initialise introduction fields }
  lblInitLetter.Caption      := Copy(FExam.Title, 1, 1);
  lblTitle.Caption           := Copy(FExam.Title, 2, Length(FExam.Title));
  memDescription.Lines.Text  := FExam.Description;
  lblCount.Caption           := IntToStr(FExam.QuestionCount);
  lblPass.Caption            := IntToStr(FExam.PassMark);
  memInstructions.Lines.Text := FExam.Instructions;
  { Initialise some score fields }
  lblScoreInitLetter.Caption := lblInitLetter.Caption;
  lblScoreTitle.Caption      := lblTitle.Caption;
  lblScorePass.Caption       := IntToStr(FExam.PassMark);
  { Start at the beginning }
  FCurQuestion := -1;
end;

{ Double-cick on a question to go to it }
procedure TfrmExam.lbxQuestionsDblClick(Sender: TObject);
begin
  with lbxQuestions do
  begin
    if ItemIndex = -1 then
      Exit;

    FCurQuestion := ItemIndex;
    ShowCurQuestion;
  end;
end;

{ Include an image for the question status in the listbox }
procedure TfrmExam.lbxQuestionsDrawItem(Control: TWinControl; Index: Integer;
  Rect: TRect; State: TOwnerDrawState);
begin
  with lbxQuestions do
  begin
    Canvas.FillRect(Rect);
    case Integer(Items.Objects[Index]) of
      Correct:   Canvas.Draw(Rect.Left, Rect.Top, FCorrect[True]);
      Incorrect: Canvas.Draw(Rect.Left, Rect.Top, FCorrect[False]);
    end;
    Canvas.TextOut(Rect.Left + 18, Rect.Top, Items[Index]);
  end;
end;

{ Treat RETURN as double-click }
procedure TfrmExam.lbxQuestionsKeyDown(Sender: TObject; var Key: Word;
  Shift: TShiftState);
begin
  if Key = VK_RETURN then
    lbxQuestionsDblClick(lbxQuestions);
end;

{ Process the answers provided for the current question }
procedure TfrmExam.ProcessCurQuestion;
var
  QuestionId: string;
begin
  if FCurQuestion = -1 then
    Exit;

  QuestionId := FSession.QuestionId[FCurQuestion];
  { A question can only be answered once }
  if not FSession.IsAnswered[QuestionId] then
    ProcessQuestion(FExam.QuestionById[QuestionId]);
end;

{ Process the answers provided for the specified question }
procedure TfrmExam.ProcessQuestion(Question: TQuestion);
var
  Index: Integer;
  Answer: string;
begin
  Answer := '';
  case Question.AnswerType of
    atCheckbox:
      { Find index(es) of selected checkbox(es) }
      for Index := 0 to Question.AnswerCount - 1 do
        with TCheckBox(FindComponent('cbxAnswer' + IntToStr(Index + 1))) do
          if Checked then
            Answer := Answer + ',' + IntToStr(Index);
    atRadio:
      { Find index of selected radio button }
      for Index := 0 to Question.AnswerCount - 1 do
        with TRadioButton(FindComponent('rabAnswer' + IntToStr(Index + 1))) do
          if Checked then
            Answer := Answer + ',' + IntToStr(Index);
    atText:
      Answer := ',' + edtAnswer.Text;
  end;
  if Answer <> '' then
    FSession.Answer[Question.Id] := Copy(Answer, 2, Length(Answer) - 1);
end;

{ Show the current question }
procedure TfrmExam.ShowCurQuestion;
var
  QuestionId: string;
begin
  QuestionId := FSession.QuestionId[FCurQuestion];
  ShowQuestion(FExam.QuestionById[QuestionId], FCurQuestion,
    FSession.Answer[QuestionId]);
end;

{ Present the specified question }
procedure TfrmExam.ShowQuestion(Question: TQuestion;
  CurQuestion: Integer; Response: string);
var
  Index: Integer;
  Answered, Selected: Boolean;
  Answers: string;

  { Has the user selected this checkbox/radio button? }
  function UserSelected(Index: Integer): Boolean;
  begin
    Result := (Pos(',' + IntToStr(Index) + ',', ',' + Response + ',') > 0);
  end;

begin
  pgcExam.ActivePage := tshQuestion;
  { Has the user answered this question? }
  Answered := (Response <> '');
  { Set page header }
  lblQuestion.Caption    := IntToStr(FCurQuestion + 1);
  memQuestion.Lines.Text := Question.Query;
  btnNext.SetFocus;

  { Set answer fields }
  case Question.AnswerType of
    atCheckbox:
      begin
        pgcAnswers.ActivePage := tshCheckBox;
        tshCheckBox.Enabled   := not Answered;
        for Index := 0 to Question.AnswerCount - 1 do
        begin
          Selected := Answered and UserSelected(Index);
          { Set checkbox }
          with TCheckBox(FindComponent('cbxAnswer' + IntToStr(Index + 1))) do
          begin
            Caption := Question.Answers[Index].Value;
            Checked := Selected;
            Visible := True;
          end;
          { If already answered show correctness }
          with TImage(FindComponent('imgCheckbox' + IntToStr(Index + 1))) do
          begin
            Visible := Answered and Question.Answers[Index].Correct;
            Picture.Assign(
              FCorrect[Question.Answers[Index].Correct and Selected]);
          end;
        end;
        { Set focus }
        if tshCheckBox.Enabled then
          cbxAnswer1.SetFocus;
        { Hide extranous answers }
        for Index := Question.AnswerCount to FMaxAnswers - 1 do
        begin
          TCheckBox(FindComponent('cbxAnswer' + IntToStr(Index + 1))).
            Visible := False;
          TImage(FindComponent('imgCheckbox' + IntToStr(Index + 1))).
            Visible := False;
        end;
      end;
    atRadio:
      begin
        pgcAnswers.ActivePage := tshRadio;
        tshRadio.Enabled      := not Answered;
        for Index := 0 to Question.AnswerCount - 1 do
        begin
          Selected := Answered and UserSelected(Index);
          { Set radio button }
          with TRadioButton(FindComponent('rabAnswer' + IntToStr(Index + 1))) do
          begin
            Caption := Question.Answers[Index].Value;
            Checked := Selected;
            Visible := True;
          end;
          { If already answered show correctness }
          with TImage(FindComponent('imgRadio' + IntToStr(Index + 1))) do
          begin
            Visible := Answered and Question.Answers[Index].Correct;
            Picture.Assign(
              FCorrect[Question.Answers[Index].Correct and Selected]);
          end;
        end;
        { Set focus }
        if tshRadio.Enabled then
        begin
          rabAnswer1.SetFocus;
          rabAnswer1.Checked := False;
        end;
        { Hide extranous answers }
        for Index := Question.AnswerCount to FMaxAnswers - 1 do
        begin
          TRadioButton(FindComponent('rabAnswer' + IntToStr(Index + 1))).
            Visible := False;
          TImage(FindComponent('imgRadio' + IntToStr(Index + 1))).
            Visible := False;
        end;
      end;
    atText:
      begin
        pgcAnswers.ActivePage := tshText;
        tshText.Enabled       := not Answered;
        { Set text }
        edtAnswer.Text := Response;
        { If already answered show correctness }
        imgText.Visible := Answered;
        imgText.Picture.Assign(FCorrect[Question.IsValid(Response)]);
        { Set focus }
        if tshText.Enabled then
          edtAnswer.SetFocus;
      end;
  end;

  { Show the explanation if already answered }
  pnlExplanation.Visible := Answered;
  if Answered then
  begin
    Answers := '';
    if Question.AnswerType = atText then
    begin
      { List valid text answers }
      for Index := 0 to Question.ValidAnswers.Count - 1 do
        Answers := Answers + ', ' + Question.ValidAnswers[Index];
      Answers := Accepted + ' ' + Copy(Answers, 3, Length(Answers)) + #13#10;
    end;
    memExplanation.Lines.Text := Answers + Question.Explanation;
  end;

  { Dis/enable navigation buttons }
  btnPrevious.Enabled := (FCurQuestion > 0);
end;

{ Present the test as a whole along with its scoring }
procedure TfrmExam.ShowScore;
const
  Grade: array [Boolean] of string = (Failed, Passed);
var
  Index, Result: Integer;
  QuestionId, Response: string;
begin
  pgcExam.ActivePage := tshScore;

  { Show the status of each question }
  with lbxQuestions.Items do
  begin
    BeginUpdate;
    Clear;
    for Index := 0 to FSession.QuestionCount - 1 do
    begin
      QuestionId := FSession.QuestionId[Index];
      Response   := FSession.Answer[QuestionId];
      if Response = '' then
        Result := Unanswered
      else if FExam.QuestionById[QuestionId].IsValid(Response) then
        Result := Correct
      else
        Result := Incorrect;
      AddObject(Question + ' ' + IntToStr(Index + 1), Pointer(Result));
    end;
    EndUpdate;
  end;

  { Show overall results }
  lblScore.Caption      := IntToStr(FSession.Score);
  lblGrade.Caption      := Grade[FSession.Score >= FExam.PassMark];
  lblGrade.Font.Color   := Highlight[FSession.Score >= FExam.PassMark];
  lblCorrect.Caption    := IntToStr(FSession.Correct);
  lblIncorrect.Caption  := IntToStr(FSession.Answered - FSession.Correct);
  lblUnanswered.Caption :=
    IntToStr(FExam.QuestionCount - FSession.Answered);
end;

end.
